<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Hash;
use App\Models\Admin;
use App\Models\User;
use App\Models\Game;
use App\Models\Question;
use App\Models\GameSession;
use Illuminate\Validation\Rule; // استيراد مهم للتحقق من البريد الإلكتروني


class AdminController extends Controller
{
    // عرض صفحة تسجيل الدخول
    public function showLoginForm()
    {
        return view('admin.auth.login');
    }

    // معالجة تسجيل الدخول
    public function login(Request $request)
    {
        $request->validate([
            'email' => 'required|email',
            'password' => 'required',
        ]);

        $credentials = $request->only('email', 'password');

        if (Auth::guard('admin')->attempt($credentials)) {
            $admin = Auth::guard('admin')->user();
            
            if (!$admin->isActive()) {
                Auth::guard('admin')->logout();
                return back()->withErrors(['email' => 'حسابك غير نشط. تواصل مع الإدارة.']);
            }

            $request->session()->regenerate();
            return redirect()->intended('/admin/dashboard');
        }

        return back()->withErrors([
            'email' => 'بيانات تسجيل الدخول غير صحيحة.',
        ]);
    }

    // تسجيل الخروج
    public function logout(Request $request)
    {
        Auth::guard('admin')->logout();
        $request->session()->invalidate();
        $request->session()->regenerateToken();
        
        return redirect('/admin/login');
    }

    // عرض لوحة التحكم الرئيسية
    public function dashboard()
    {
        $stats = [
            'total_users' => User::count(),
            'total_games' => Game::count(),
            'total_questions' => Question::count(),
            'total_sessions' => GameSession::count(),
            'completed_sessions' => GameSession::where('status', 'completed')->count(),
            'recent_users' => User::latest()->take(5)->get(),
            'recent_sessions' => GameSession::with(['user', 'game'])
                ->latest()
                ->take(10)
                ->get(),
        ];

        return view('admin.dashboard', compact('stats'));
    }

    // عرض قائمة المديرين
    public function admins()
    {
        $admins = Admin::paginate(20);
        return view('admin.admins.index', compact('admins'));
    }

    // عرض صفحة إنشاء مدير جديد
    public function createAdmin()
    {
        return view('admin.admins.create');
    }

    // حفظ مدير جديد
    public function storeAdmin(Request $request)
    {
        $request->validate([
            'name' => 'required|string|max:255',
            'email' => 'required|string|email|max:255|unique:admins',
            'password' => 'required|string|min:8|confirmed',
            'role' => 'required|in:admin,super_admin',
        ]);

        Admin::create([
            'name' => $request->name,
            'email' => $request->email,
            'password' => Hash::make($request->password),
            'role' => $request->role,
        ]);

        return redirect()->route('admin.admins.index')
            ->with('success', 'تم إنشاء المدير بنجاح');
    }

    // تعديل حالة المدير (تفعيل/إلغاء تفعيل)
    public function toggleAdminStatus(Admin $admin)
    {
        $admin->update(['is_active' => !$admin->is_active]);
        
        $status = $admin->is_active ? 'تم تفعيل' : 'تم إلغاء تفعيل';
        
        return back()->with('success', "$status المدير بنجاح");
    }

 /**
     * ✅ دالة جديدة: عرض صفحة الملف الشخصي للمدير.
     */
    public function showProfileForm()
    {
        // جلب بيانات المدير المسجل حالياً
        $admin = Auth::guard('admin')->user();
        return view('admin.profile', compact('admin'));
    }

    /**
     * ✅ دالة جديدة: تحديث بيانات الملف الشخصي للمدير.
     */
    public function updateProfile(Request $request)
    {
        // جلب المدير المسجل حالياً
        $admin = Auth::guard('admin')->user();

        // التحقق من صحة البيانات المدخلة
        $validatedData = $request->validate([
            'name' => 'required|string|max:255',
            // البريد الإلكتروني يجب أن يكون فريداً، باستثناء البريد الخاص بالمدير الحالي
            'email' => [
                'required',
                'string',
                'email',
                'max:255',
                Rule::unique('admins')->ignore($admin->id),
            ],
            // كلمة المرور اختيارية، وإذا تم إدخالها يجب أن تكون مؤكدة
            'password' => 'nullable|string|min:8|confirmed',
        ]);

        // تحديث الاسم والبريد الإلكتروني
        $admin->name = $validatedData['name'];
        $admin->email = $validatedData['email'];

        // إذا قام المدير بإدخال كلمة مرور جديدة، قم بتشفيرها وتحديثها
        if (!empty($validatedData['password'])) {
            $admin->password = Hash::make($validatedData['password']);
        }

        // حفظ كل التغييرات
        $admin->save();

        return redirect()->route('admin.profile.show')->with('success', 'تم تحديث ملفك الشخصي بنجاح.');
    }

 /**
     * ✅ دالة جديدة: عرض صفحة تعديل مدير.
     */
    public function editAdmin(Admin $admin)
    {
        return view('admin.admins.edit', compact('admin'));
    }

    /**
     * ✅ دالة جديدة: تحديث بيانات مدير موجود.
     */
    public function updateAdmin(Request $request, Admin $admin)
    {
        $validatedData = $request->validate([
            'name' => 'required|string|max:255',
            'email' => ['required', 'string', 'email', 'max:255', Rule::unique('admins')->ignore($admin->id)],
            'role' => 'required|in:admin,super_admin',
            'password' => 'nullable|string|min:8|confirmed',
        ]);

        $admin->name = $validatedData['name'];
        $admin->email = $validatedData['email'];
        $admin->role = $validatedData['role'];

        if (!empty($validatedData['password'])) {
            $admin->password = Hash::make($validatedData['password']);
        }

        $admin->save();

        return redirect()->route('admin.admins.index')->with('success', 'تم تحديث بيانات المدير بنجاح.');
    }

    /**
     * ✅ دالة جديدة: حذف مدير.
     */
    public function destroyAdmin(Admin $admin)
    {
        // إجراء أمان: لا تسمح للمدير بحذف حسابه الشخصي
        if ($admin->id === Auth::guard('admin')->id()) {
            return back()->with('error', 'لا يمكنك حذف حسابك الشخصي.');
        }

        // إجراء أمان آخر: لا تسمح بحذف آخر مدير خارق
        if ($admin->role === 'super_admin' && Admin::where('role', 'super_admin')->count() === 1) {
            return back()->with('error', 'لا يمكن حذف آخر مدير خارق في النظام.');
        }

        $admin->delete();

        return redirect()->route('admin.admins.index')->with('success', 'تم حذف المدير بنجاح.');
    }

}

